using System;
using System.Collections.Concurrent;
using System.Linq;
using System.Runtime.Caching;

namespace HIPS.Web.Components.Cache
{
    /// <summary>
    /// Provides a memory-based cache.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
    /// </history>
    public class MemoryCacheProvider : CacheProvider
    {

        #region Fields

        /// <summary>
        /// Used for double-checked locking.
        /// This is static as all instances refer to MemoryCache.Default
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        private static readonly ConcurrentDictionary<string, object> CacheLock = new ConcurrentDictionary<string, object>();

        #endregion

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="defaultAbsoluteExpirationOffset">The default absolute expiration offset, or null if not using absolute expiration.</param>
        /// <param name="defaultSlidingExpiration">The default sliding expiration, or null if not using sliding expiration.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        public MemoryCacheProvider(TimeSpan? defaultAbsoluteExpirationOffset)
            : base(defaultAbsoluteExpirationOffset, null)
        {
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="defaultAbsoluteExpirationOffset">The default absolute expiration offset, or null if not using absolute expiration.</param>
        /// <param name="defaultSlidingExpiration">The default sliding expiration, or null if not using sliding expiration.</param>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        public MemoryCacheProvider(TimeSpan? defaultAbsoluteExpirationOffset, TimeSpan? defaultSlidingExpiration)
            : base(defaultAbsoluteExpirationOffset, defaultSlidingExpiration)
        {
        }

        #endregion

        #region Methods
        
        /// <summary>
        /// Gets the value identified by the cache key from the cache.
        /// Returns null if there is no cached value for the specified key.
        /// </summary>
        /// <typeparam name="T">Type of the item to be retrieved from the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <returns>
        /// The cached value for the specified key, or null if no value found in the cache.
        /// </returns>
        public override T Get<T>(string cacheKey)
        {
            return MemoryCache.Default.Get(cacheKey) as T;
        }

        /// <summary>
        /// Stores the provided value against the identified cache key.
        /// If the provided value is null the identified cache key is cleared.
        /// Dependencies are supported.
        /// </summary>
        /// <typeparam name="T">Type of the item to be stored in the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="item">The item value to set. If null is provided the key is cleared.</param>
        /// <param name="slidingExpiration">A value indicating when a cache entry should be evicted if it has not been accessed in a given span of time.</param>
        /// <param name="absoluteExpiration">A value indicating whether a cache entry should be evicted at a specified date and time.</param>
        /// <param name="dependencyKeys">An optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        public override void Set<T>(string cacheKey, T item, TimeSpan slidingExpiration, DateTimeOffset absoluteExpiration, params string[] dependencyKeys)
        {
            // Null values cannot be cached and are interpreted as clearing the cache.
            if (item == null)
            {
                // Remove is thread-safe and does not throw if the key doesn't exist.
                MemoryCache.Default.Remove(cacheKey);
                return;
            }

            /* Set cache key to equal item */

            // Expiry Policy
            CacheItemPolicy policy = new CacheItemPolicy() { AbsoluteExpiration = absoluteExpiration, SlidingExpiration = slidingExpiration };
            
            // Support for Dependency Keys
            if (dependencyKeys != null && dependencyKeys.Any())
            {
                policy.ChangeMonitors.Add(MemoryCache.Default.CreateCacheEntryChangeMonitor(dependencyKeys));
            }

            // Set is thread-safe and does not throw if the key already exists.
            MemoryCache.Default.Set(cacheKey, item, policy);
        }

        /// <summary>
        /// Returns a reference to an object that can be used as an applicable thread lock for this cache for the specified key.
        /// This is useful for functionality such as double-check locking. 
        ///  </summary>
        /// <param name="cacheKey">The cache key.</param>
        /// <returns>A cache-specific key-specific thread lock.</returns>
        public override object GetLock(string cacheKey)
        {
            return CacheLock.GetOrAdd(cacheKey, new object());
        }

        #endregion
    }
}